/*
 * Dump info about a gif file
 *
 * Copyright (C) 2014 Roger Walker - smallgif.com
 * Commercial and Non-commercial is allowed, for this or it's derivatives providing you credit me
 * In the "About box" or documentation you should mention "The gif image file functions are based on work by Roger@smallgif.com"
 */
const char *version = "_$Id: gifinfo.c,v 1.17 2013/03/21 02:26:11 roger Exp $_";

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <float.h>
#include <math.h>

unsigned guVerboseLevel=1;
#define GIF_LOAD_STATS guVerboseLevel
#include "gif_load.h"

unsigned guImageLevel=0;
unsigned guColourLevel=0;

static void dump_colours(unsigned const uNColours,gif_colour_entry_t const *psColours)
{
	if (guImageLevel==1 || guImageLevel==2)
	{	unsigned uColour;
		printf("%u colours:-\n",uNColours);
		for (uColour=0 ; uColour<uNColours ; uColour++)
		{	if ((uColour%8)==0) printf ("%02X=",uColour);
			printf("%02X%02X%02X%s",
				psColours[uColour].red,psColours[uColour].green,psColours[uColour].blue,
				((uColour%8)==7 || uColour==(uNColours-1)) ? "\n" : " ");
		}
	}
}


int main(int argc,char *argv[])
{	gif_file_t sSrcGif;
	unsigned uImage;
	const gif_image_t *psImage;
	unsigned uFile=0;

	{	unsigned uArg;
		for (uArg=0 ; uArg<(unsigned)argc ; uArg++)
		if (strnicmp(argv[uArg],"/v",2)==0 || strnicmp(argv[uArg],"-v",2)==0)
		{	if (isdigit(argv[uArg][2]))
			{	guVerboseLevel=atoi(argv[uArg]+2);
			} else
			{	unsigned uOpt;
				for (uOpt=1 ; uOpt<strlen(argv[uArg]) ; uOpt++)
					if (toupper(argv[uArg][uOpt])=='V') guVerboseLevel++;
			}
		} else
		if (strnicmp(argv[uArg],"/i",2)==0 || strnicmp(argv[uArg],"-i",2)==0)
		{	if (isdigit(argv[uArg][2]))
			{	guImageLevel=atoi(argv[uArg]+2);
			} else
			{	unsigned uOpt;
				for (uOpt=1 ; uOpt<strlen(argv[uArg]) ; uOpt++)
					if (toupper(argv[uArg][uOpt])=='I') guImageLevel++;
			}
		} else
		if (strnicmp(argv[uArg],"/c",2)==0 || strnicmp(argv[uArg],"-c",2)==0)
		{	if (isdigit(argv[uArg][2]))
			{	guColourLevel=atoi(argv[uArg]+2);
			} else
			{	unsigned uOpt;
				for (uOpt=1 ; uOpt<strlen(argv[uArg]) ; uOpt++)
					if (toupper(argv[uArg][uOpt])=='C') guColourLevel++;
			}
		} else
		if (uFile==0)
		{	uFile=uArg;
		} else
		{	// Error
			uFile=0;
			break;
		}
	}

	if (uFile==0)
	{	fprintf(stderr,"SYNTAX: gifinfo [<-v>...] [<-i>...] [<-c>...]"
					"<filename>\n"
				"-v\tVerbose information\n"
				"-i\tImage information detail\n"
				"-c\tColour information detail\n"
				);
		return 99;
	}

	if (gif_read_file(&sSrcGif,argv[uFile]))
	{	fprintf(stderr,"Error: Problem loading source image\n");
		return 10;
	}

	if (sSrcGif.uNImages<1)
	{	fprintf(stderr,"Error: No image in source file\n");
		gif_free(&sSrcGif);
		return 10;
	}

	printf("\nScreen info:");
	printf("%u x %u pixels, %u bits per pixel\n",
		sSrcGif.screen.width,sSrcGif.screen.height,sSrcGif.screen.bitsPerPixelMinus1+1);
	if (sSrcGif.sOptions.byCompressionLevel!=0xFF) printf("Compressed with mode %u\n",sSrcGif.sOptions.byCompressionLevel);
	if (!sSrcGif.sOptions.bNoOverrun) printf("Had compressed block overruns\n");
	if (sSrcGif.sOptions.bMissingInitialResets) printf("Had missing initial resets\n");
	printf("%u significant bits per colour value\n",sSrcGif.screen.bitsPerColourValMinus1+1);
	if (sSrcGif.screen.sort) printf("Colours are sorted\n");
	if (sSrcGif.screen.aspect) printf("Aspect ratio %u : 64",sSrcGif.screen.aspect+15);

	if (sSrcGif.screen.bGlobalColourTable)
	{	printf("Background colour = %u\n",sSrcGif.screen.bgColour);

		if (guImageLevel==1 || guImageLevel==2 || guColourLevel==2)
		{	printf("Global colour table with ");
			dump_colours(GIF_NCOLOURS(sSrcGif.screen),sSrcGif.psColourTable);
		} else
		{	printf("Global colour table with %u colours\n",GIF_NCOLOURS(sSrcGif.screen));
		}
	}

	for (uImage=0,psImage=sSrcGif.psImages ; uImage<sSrcGif.uNImages ; uImage++,psImage++)
	{	printf("Image %u:\n",uImage);
		if (psImage->bHaveControl)
		{	const gif_ext_graphics_control_label_t *pCtrl=&psImage->control;
			printf("Control block:-\n");
			if (pCtrl->bTransparent) printf("Transparent colour %u\n",pCtrl->byTransparentColour);
			if (pCtrl->bUserinput)	printf("Pause for user input\n");
			printf("Disposal method = %u = %s\n",pCtrl->disposal,
				(pCtrl->disposal==0) ? "Nothing" :
				(pCtrl->disposal==1) ? "Leave Image" :
				(pCtrl->disposal==2) ? "Set background Colour" :
				(pCtrl->disposal==3) ? "Restore Previous frame" :  "Unknown");
			if (pCtrl->delaytime) printf("Delay for %u/100 sec\n",pCtrl->delaytime);
		}

		printf("Data: (%u,%u)+(%u,%u)=(%u,%u)",
			psImage->image.left,psImage->image.top,psImage->image.width,psImage->image.height,
			psImage->image.left+psImage->image.width,psImage->image.top+psImage->image.height);
		if (psImage->image.bInterlaced)
		{	printf(" (Interlaced)");
		}
		if (psImage->image.bLocalColourTable)
		{	assert(psImage->psColourTable);
			printf(" %u local colours\n",GIF_NCOLOURS(psImage->image));
		} else
		{	assert(sSrcGif.screen.bGlobalColourTable);
			printf(" using global colours\n");
		}
		if (psImage->image.bLocalColourTable)
		{	dump_colours(GIF_NCOLOURS(psImage->image),psImage->psColourTable);
		}

		if (psImage->bHaveNetscapeLoop)
		{	printf("Repeat for %u loops\n",psImage->wNumLoops);
		}

		if (guImageLevel>=2)
		{	unsigned const uNColours=(psImage->image.bLocalColourTable) ? GIF_NCOLOURS(psImage->image) : GIF_NCOLOURS(sSrcGif.screen);
			unsigned uRow,uCol;
			for (uRow=0 ; uRow<psImage->image.height ; uRow++)
			{	printf("%3u:",uRow);
				for (uCol=0 ; uCol<psImage->image.width ; uCol++)
					printf((uNColours<=16) ? " %X" : " %02X",psImage->pLines[uRow][uCol]);
				printf("\n");
			}
		}

		if (guColourLevel)
		{	unsigned uPixel;
			unsigned auColourUsed[256];
			unsigned uUsed=0;
			BYTE *pbyPix;
			memset(auColourUsed,0,sizeof(auColourUsed));
			pbyPix= &psImage->pLines[0][0];
			for (uPixel=0 ; uPixel<psImage->uNPixels ; uPixel++,pbyPix++)
			{	if (auColourUsed[*pbyPix]==0)
					uUsed++;
				auColourUsed[*pbyPix]++;
			}
			printf("%u colours used\n",uUsed);
			if (guColourLevel>=2)
			{	unsigned uColour;
				gif_colour_entry_t const *psColours=(psImage->image.bLocalColourTable) ? psImage->psColourTable : sSrcGif.psColourTable;
				for (uColour=0 ; uColour<256 ; uColour++,psColours++)
				if (auColourUsed[uColour])
				{	printf("%3u: %02X %02X %02X : %u used\n",
						uColour,psColours->red,psColours->green,psColours->blue,auColourUsed[uColour]);
				}
			}
		}
	}
	gif_free(&sSrcGif);
	return 0;
}
