#ifndef DONE_GIF_LS_H	// [
#define DONE_GIF_LS_H	"_$Id: gif_ls.h,v 1.146 2014/01/16 14:31:00 roger Exp $_"

/*
 * GIF file save routines
 *
 * Copyright (C) 2014 Roger Walker - smallgif.com
 * Commercial and Non-commercial is allowed, for this or it's derivatives providing you credit me
 * In the "About box" or documentation you should mention "The gif image file functions are based on work by Roger@smallgif.com"
 */
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include "gif_load.h"

#ifdef GIF_LS_STATS
#include <time.h>
#endif

#ifdef __cplusplus
extern "C" {
#endif

/********************** Public function definitions *************************/

extern void gif_clear_unused_colours(gif_colour_entry_t *psColours,unsigned const uNColours,bool const *pbUsedList);
extern bool gif_write_images(gif_file_t const *psGifInfo,gif_options_t const *psOptions,
					bool	(*pfnWriteData)(const BYTE *pDest,unsigned uNBytes,void *pvDataHandle),
					void	*pvDataHandle);
extern bool gif_write_file(gif_file_t const *pInfo,gif_options_t const *psOptions,char const *pFileName);
extern bool gif_copy_file_info(gif_file_t *pDest,gif_file_t const *pSrc);
#ifdef GIF_PRIVATE_WRITE_TO_MEM
extern bool gif_write_memory(const gif_file_t *pInfo,gif_options_t const *psOptions,char **pData,unsigned *puLength);
#endif

/*************************** Private function definitions **************************/
#ifndef NO_FUNCTIONS	// [

#ifdef GIF_ASSERT
#undef GIF_ASSERT
#endif
#if defined(GIF_LS_DEBUG) || defined(_DEBUG)
#define GIF_PRIVATE_LS_MAGIC 0xBEEFDEAD
#define GIF_ASSERT(a) assert(a)
#else
#define GIF_ASSERT(a)
#endif

/* Private definitions */

// Optional status information
typedef struct gif_private_write_status_tag
{	unsigned long uBitsUsed;	/* Current number of bits written */
	unsigned uMaxOverrun;		/* Max overruns used */
} gif_private_write_status_t;

// Optional ratio information
typedef struct gif_private_write_ratio_info_tag
{	unsigned uEndPixel;
	unsigned uEndBit;
#ifdef GIF_PRIVATE_LS_MAGIC
	unsigned uMagic;
#endif
} gif_private_write_ratio_info_t;

struct gif_private_write_comp_state_tag;	// Forward definition of main structure

// Data that is based on the file's information
typedef struct gif_private_write_data_tag
{	// Pointer to image bytes
	BYTE const *pbyPixels;
	unsigned uNPixels;

	BYTE	byBitsPerPixel;

	// Compression options
	gif_options_t sOptions;

#ifndef GIF_LS_NOSTATUS
	// Use for status display
	unsigned uPixelsTotal;
	unsigned uPixelsDone;
#endif

	// Data handle passed directly via call for byte writing, or accessable via pointers for code writes
	void	*pvDataHandle;
	// Writing bytes via this function pointer
	bool	(*pfnWriteData)(const BYTE *pSrc,unsigned uNBytes,void *pvDataHandle);
	// Writing codes, via this function pointer
	bool	(*pfnWriteCode)(struct gif_private_write_comp_state_tag *psInfo, gif_code_t code);

	// If non-NULL then track info
	gif_private_write_status_t *psStatus;
} gif_private_write_data_t;

// Data that is setup before the file write, and is then constant
typedef struct gif_private_write_info_tag
{	// Used during the file writing, selected based on available memory
#ifdef GIF_LS_DEBUG
	bool	bLowMemoryOption;			/* Low memory will use search, otherwise direct lookup */
#endif
	void	(*pfnCodeReset)(struct gif_private_write_comp_state_tag *psInfo);
	bool	(*pfnCodeFind)(struct gif_private_write_comp_state_tag *psInfo,gif_code_t cNewCode);
	void	(*pfnCodeAdd)(struct gif_private_write_comp_state_tag *psInfo);
	void	(*pfnSearchFree)(struct gif_private_write_comp_state_tag *psCompState,struct gif_private_write_info_tag *psCompInfo);

	BYTE	byInitCodeSize;				/* Initial code table width in bits */
	BYTE	byInitialBitsPerCode;		/* Initial number of bits per code, for this file */
} gif_private_write_info_t;

// Current compression state
typedef struct gif_private_write_comp_state_tag
{	// Pointer to parent compression structure
	gif_private_write_data_t const *psCompData;
	gif_private_write_info_t const *psCompInfo;

	unsigned	uStartPixel;	/* Startpixel for this block */
	unsigned	uPixel;			/* Current pixel pos */

	void	*pvDataLookup;		/* Private struture pointer for pfnCodeReset,pfnCodeFind & pfnCodeAdd functions */
	void	*pvDecisionData;	/* Private structure pointer for pfnDecision function */

	gif_code_t	cSpecial;		/* Start of the special codes, +0 = Clear, +1 = EOI */
	gif_code_t	cNextEntry;		/* Next unused entry */
	gif_code_t	cPrevious;		/* The previous code */

	BYTE byNCodeBits;			/* number of bits/code */
	gif_code_t	cMax;			/* maximum code, given byNCodeBits */

	unsigned uExtraPixels;		/* Keep track of current token overrun length */

	unsigned long uBitsWritten;	/* Total number of bits written */

	// bit buffer, has to handle 7+GIF_LZ_BITS = 7+12 = 19 bits
	BYTE byBitBufferNBits;		/* Number of bits still valid in the ulBitBuffer */
	unsigned long ulBitBuffer;	/* ulBitBuffer, contains partial byte */

	// Byte buffer
	BYTE	abyBuf[256];		/* Compressed data is buffered here., buf[0]=Number of bytes in buffer (0..255) */
} gif_private_write_comp_state_t;

// Major compression table state manipulation functions
static bool gif_private_write_comp_init(gif_private_write_comp_state_t *psCompState,gif_private_write_info_t *psCompInfo,gif_private_write_data_t const *psCompData,bool (*pfnSearchInit)(gif_private_write_comp_state_t *psCompState,gif_private_write_info_t *psCompInfo));
static void gif_private_write_comp_reset(gif_private_write_comp_state_t *psCompState);
static bool gif_private_write_comp_clearcode(gif_private_write_comp_state_t *psCompState);
static bool gif_private_write_comp_eoi(gif_private_write_comp_state_t *psCompState);
static bool gif_private_search_init(gif_private_write_comp_state_t *psCompState,gif_private_write_info_t *psCompInfo);

#ifdef GIF_PRIVATE_WRITE_TO_MEM
// Internal structure, used when writing data bytes to a block of memory
typedef struct
{	BYTE *pData;
	unsigned uLen;
	unsigned uAlloced;
} gif_private_write_mem_t;

// Write bytes to a memory block, void *pvDataHandle==gif_private_write_mem_t * (returns true on error)
static bool gif_private_write_data_to_mem(const BYTE *pSrc,unsigned uNBytes,void *pvDataHandle)
{	gif_private_write_mem_t *pMemInfo=(gif_private_write_mem_t *)pvDataHandle;
	unsigned const uReallocBlockSize=16384;	// Must be power of 2
	if ((pMemInfo->uLen+uNBytes)>pMemInfo->uAlloced)
	{	pMemInfo->uAlloced=((pMemInfo->uLen+uNBytes)&(~(uReallocBlockSize-1)))+uReallocBlockSize;
		if (gif_realloc(&pMemInfo->pData,pMemInfo->uAlloced))
		{	fprintf(stderr,"Error: Failed to realloc %u+%u bytes\n",pMemInfo->uLen,uNBytes);
			return true;
		}
	}
	GIF_ASSERT((pMemInfo->uLen+uNBytes)<=pMemInfo->uAlloced);
	memcpy(pMemInfo->pData+pMemInfo->uLen,pSrc,uNBytes);
	pMemInfo->uLen+=uNBytes;
	return false;
}
#endif	// GIF_PRIVATE_WRITE_TO_MEM

/*
 * Gif Writing functions
 */

// Write a byte
static bool gif_private_write_byte(gif_private_write_data_t const *psCompData, BYTE b)
{	GIF_ASSERT(psCompData->pfnWriteData!=NULL);
	return psCompData->pfnWriteData((BYTE *)&b,1,psCompData->pvDataHandle);
}

// Write a word
static bool gif_private_write_word(gif_private_write_data_t const *psCompData, WORD w)
{	BYTE abyBuf[2];
	GIF_ASSERT(psCompData->pfnWriteData!=NULL);
	// Avoid endian problems by storing word value into bytes
	abyBuf[0]=(BYTE)(w&0xFF);
	abyBuf[1]=(BYTE)((w>>8)&0xFF);
	// Save bytes
	return psCompData->pfnWriteData(abyBuf,2,psCompData->pvDataHandle);
}

// Reset the block buffer length
static void gif_private_write_buffer_init(gif_private_write_comp_state_t *psCompState)
{	GIF_ASSERT(psCompState);
	psCompState->abyBuf[0] = 0;	// abyBuf[0]=Length
}

// Flush block buffer to output
static bool gif_private_write_buffer_flush(gif_private_write_comp_state_t *psCompState)
{	bool bRetVal;
	GIF_ASSERT(psCompState->psCompData->pfnWriteData!=NULL);
	bRetVal=psCompState->psCompData->pfnWriteData(psCompState->abyBuf,psCompState->abyBuf[0]+1,psCompState->psCompData->pvDataHandle);
#if defined(GIF_LS_DEBUG) && GIF_LS_DEBUG>=6
	printf("W%02X",psCompState->abyBuf[0]);
	{	unsigned uByte;
		for (uByte=1 ; uByte<=psCompState->abyBuf[0] ; uByte++)
		{	printf(" %02X",psCompState->abyBuf[uByte]);
		}
		printf("\n");
	}
	memset(psCompState->abyBuf,0xAA,sizeof(psCompState->abyBuf));
#endif
	psCompState->abyBuf[0]=0;
	return bRetVal;
}

// Add byte to block buffer, if it's reach limit flush it and continue
static bool gif_private_write_buffer_add_byte(gif_private_write_comp_state_t *psCompState,BYTE byData)
{	GIF_ASSERT(psCompState->psCompData->pfnWriteData!=NULL);
	psCompState->abyBuf[ ++psCompState->abyBuf[0] ] = byData;
	if( psCompState->abyBuf[0] < 255 ) return false;
	return gif_private_write_buffer_flush(psCompState);
}

#define GIF_PRIVATE_MAXCODE(uCodeBits)	(WORD)(( 1U << (uCodeBits)) - 1U)

/*
 * Write a code value, as psCompState->byNCodeBits to the bitbuffer/byte buffer
 * Otheriwse if we reach the code length grow it
 */
static bool gif_private_write_code(gif_private_write_comp_state_t *psCompState, gif_code_t  code)
{	// Sanity check byNCodeBits range, and code value
	GIF_ASSERT(psCompState->byNCodeBits>=3);	// 1bpp + 2 initial code size = 3
	GIF_ASSERT(psCompState->byNCodeBits<=GIF_LZ_BITS);
	GIF_ASSERT(code<(1U<<psCompState->byNCodeBits));

	// Check we've got a write function, or at least monitoring status, otherwise no reason to get this far
	GIF_ASSERT(psCompState->psCompData->pfnWriteData!=NULL || psCompState->psCompData->pfnWriteCode!=NULL || psCompState->psCompData->psStatus);

	// Add the bits to the total bit count
	psCompState->uBitsWritten +=psCompState->byNCodeBits;

	// If we've got a code write function, call it
	if (psCompState->psCompData->pfnWriteCode!=NULL)
	{	psCompState->psCompData->pfnWriteCode(psCompState,code);
	}

#if defined(GIF_LS_DEBUG) && GIF_LS_DEBUG>=4
#if GIF_LS_DEBUG>=5
	printf("Out:");
#endif
	printf(" %03X",code);
	if (code==(gif_code_t)(psCompState->cSpecial+0))	// cSpecial+0 == Clear
	{	printf(" (CLR)");
	} else
	if (code==(gif_code_t)(psCompState->cSpecial+1))	// cSpecial+1 == EOI
	{	printf(" (EOI)");
	}
#if GIF_LS_DEBUG>=5
	printf(" (+%u=%u)\n",psCompState->byNCodeBits,psCompState->uBitsWritten);
#endif
#endif

	// If we've got a byte write function, sort out the bit buffer, and call the byte buffer function
	if (psCompState->psCompData->pfnWriteData!=NULL)
	{	// If we've got an output function, buffer the bits
		GIF_ASSERT(psCompState->byBitBufferNBits<8); // A byte or more should have been saved by now
		GIF_ASSERT(psCompState->ulBitBuffer<(1U<<psCompState->byBitBufferNBits));	// Top bits should be empty (we always shift 0's in)
		if( psCompState->byBitBufferNBits > 0 )
		{	// Merge in new bits to top of bitstream
			psCompState->ulBitBuffer |= ((long)code << psCompState->byBitBufferNBits);
		} else
		{	// Empty buffer, so just set it to value
			psCompState->ulBitBuffer = code;
		}

		// Include new bits in count
		psCompState->byBitBufferNBits = (BYTE)(psCompState->byBitBufferNBits+psCompState->byNCodeBits);
		GIF_ASSERT(psCompState->byBitBufferNBits<=(7+GIF_LZ_BITS));

		// While the bit buffer is bigger than a byte's worth, output bytes
		while( psCompState->byBitBufferNBits >= 8 )
		{	if (gif_private_write_buffer_add_byte(psCompState, (BYTE)(psCompState->ulBitBuffer & 0xff) ))
				return true;
			psCompState->ulBitBuffer >>= 8;
			psCompState->byBitBufferNBits -= 8;
		}
	}

	// If it's a normal code, check code size
	if (code <(int)psCompState->cSpecial || code >=(int)(psCompState->cSpecial+2))
	{	// Just a normal code
		GIF_ASSERT(psCompState->cNextEntry<=GIF_LZ_NCODES);
		if ( psCompState->cNextEntry > psCompState->cMax)
		{	// The next entry is going to be too big for the code size, so increase code size
			psCompState->byNCodeBits++;
			if ( psCompState->byNCodeBits>=GIF_LZ_BITS)
			{	// If we're at the max, allow overrun codes (we might of already been here)
				psCompState->cMax = GIF_LZ_NCODES;
			} else
			{	// Set the next code step point
				psCompState->cMax = GIF_PRIVATE_MAXCODE(psCompState->byNCodeBits);
			}
		}
	}
	return false;
}

#ifdef GIF_LS_DEBUG
/*
 * Use direct 2D array lookup of the prefix code / next character combinations.
 * In low memory use a brute force search of the hash table
 * Use an adaptive reset, where the code table is cleared when the compression ratio decreases, rather than when after the table fills.
 */
#endif

// Reset the code tables
static void gif_private_write_comp_reset(gif_private_write_comp_state_t *psCompState)
{	GIF_ASSERT(psCompState && psCompState->psCompData);

	// Calc codes
	psCompState->byNCodeBits = psCompState->psCompInfo->byInitialBitsPerCode;
	psCompState->cMax = GIF_PRIVATE_MAXCODE(psCompState->byNCodeBits);
	psCompState->cSpecial = (gif_code_t)(1U << (psCompState->psCompInfo->byInitialBitsPerCode - 1));
	psCompState->cNextEntry = (gif_code_t)(psCompState->cSpecial + 2);	// +0=Clear, +1=EOI, +2=First usable code
	psCompState->psCompInfo->pfnCodeReset(psCompState);
}

// Initialise code tables
static bool gif_private_write_comp_init(gif_private_write_comp_state_t *psCompState,gif_private_write_info_t *psCompInfo,gif_private_write_data_t const *psCompData,bool (*pfnSearchInit)(gif_private_write_comp_state_t *psCompState,gif_private_write_info_t *psCompInfo))
{	GIF_ASSERT(psCompState && psCompData && psCompInfo);

	memset(psCompState,0,sizeof(*psCompState));
	memset(psCompInfo,0,sizeof(*psCompInfo));
	psCompState->psCompData = psCompData;
	psCompState->psCompInfo = psCompInfo;

	// Sort out initial code table values
	psCompInfo->byInitCodeSize = (BYTE)(( psCompData->byBitsPerPixel<= 2 ) ? 2 : psCompData->byBitsPerPixel);
	psCompInfo->byInitialBitsPerCode = (BYTE)(psCompInfo->byInitCodeSize+1);

	if (pfnSearchInit(psCompState,psCompInfo))
		return true;

	// Not written any data yet
	psCompState->uBitsWritten =0;

	// Clear the bit buffer
	psCompState->ulBitBuffer = 0;
	psCompState->byBitBufferNBits = 0;

	// Initialise the write buffer
	gif_private_write_buffer_init(psCompState);

	// Reset tables
	gif_private_write_comp_reset(psCompState);
	return false;
}

// Send a clear code, and reset the code tables
static bool gif_private_write_comp_clearcode(gif_private_write_comp_state_t *psCompState)
{	// Output clear code
	if (gif_private_write_code(psCompState, (gif_code_t)(psCompState->cSpecial+0)))	// cSpecial+0 == Clear code
		return true;

	gif_private_write_comp_reset(psCompState);
	return false;
}

// Write the end of image marker, then flush the data buffers
static bool gif_private_write_comp_eoi(gif_private_write_comp_state_t *psCompState)
{	// Send the EOI
	if (gif_private_write_code(psCompState, (gif_code_t)(psCompState->cSpecial+1)))	// cSpecial+1 == End of Image
		return true;

	// If we've an output function, flush the buffers
	if (psCompState->psCompData->pfnWriteData!=NULL)
	{	// Time to flush the buffers

		// If there are bits in the bit buffer, flush them
		if (psCompState->byBitBufferNBits > 0 )
		{	GIF_ASSERT(psCompState->byBitBufferNBits<8);
			if (gif_private_write_buffer_add_byte(psCompState, (BYTE)(psCompState->ulBitBuffer & 0xff) ))
				return true;
			psCompState->byBitBufferNBits=0;
		}

		// If there are bytes in the block buffer, flush it
		if (psCompState->abyBuf[0]!=0 && gif_private_write_buffer_flush(psCompState))
				return true;
	}

	return false;
}

static void gif_private_search_free(gif_private_write_comp_state_t *psCompState,gif_private_write_info_t *psCompInfo)
{	if (psCompInfo->pfnSearchFree)
		psCompInfo->pfnSearchFree(psCompState,psCompInfo);
}

typedef struct gif_private_search_tag
{	// For failed search, remember byte code
	BYTE				byAddPixel;
	gif_code_t	cAddCode;
	// Table of code pairs
	gif_code_t	acChainedCode[GIF_LZ_NCODES];
	BYTE				abyChainedPixel[GIF_LZ_NCODES];
} gif_private_search_t;

static void gif_private_search_reset(gif_private_write_comp_state_t *psCompState)
{	gif_private_search_t * const psInfo=psCompState->pvDataLookup;
#ifdef GIF_LS_DEBUG
	GIF_ASSERT(psCompState->psCompInfo->bLowMemoryOption);	// Low memory will use search, otherwise direct lookup
#endif
	// Clear tables
	memset(psInfo->acChainedCode,0xFF,sizeof(psInfo->acChainedCode));
	memset(psInfo->abyChainedPixel,0xFF,sizeof(psInfo->abyChainedPixel));
}

// Look for value pair, returns true if value found, otherwise false
static bool gif_private_search_find(gif_private_write_comp_state_t *psCompState,gif_code_t cNewPixel)
{	gif_private_search_t * const psInfo=psCompState->pvDataLookup;
	unsigned uSearch;

#ifdef GIF_LS_DEBUG
	GIF_ASSERT(psCompState->psCompInfo->bLowMemoryOption);	// Low memory will search, otherwise direct lookup
#endif

	// Sanity check inputs
	GIF_ASSERT(psCompState && psCompState->psCompInfo && psCompState->psCompData);
	GIF_ASSERT(psCompState->cPrevious<GIF_LZ_NCODES);
	GIF_ASSERT(cNewPixel<(1U<<psCompState->psCompData->byBitsPerPixel));
	GIF_ASSERT(psCompState->cNextEntry>=(psCompState->cSpecial+2));

	// Search for this entry
	for (uSearch=psCompState->cSpecial+2 ; uSearch<GIF_LZ_NCODES ; uSearch++)
	{	// Check if end of valid codes?
		if (psInfo->acChainedCode[uSearch]>GIF_LZ_NCODES)
		{	// End of table, not found
			GIF_ASSERT(uSearch==psCompState->cNextEntry);
			break;
		}
		// Check for the code we're searching for
		if (psInfo->abyChainedPixel[uSearch]==((BYTE)cNewPixel) &&
			psInfo->acChainedCode[uSearch]==psCompState->cPrevious)
		{	// Yep - found it
#if defined(GIF_LS_DEBUG) && GIF_LS_DEBUG>=4
			{	gif_code_t	codeSearch2;
				gif_code_t	codeThis,cNextEntry;
				printf("Chaining %03X followed by %02X",psCompState->cPrevious,cNewPixel);
				codeThis=psCompState->cPrevious;
				cNextEntry=cNewPixel;
				do
				{	for (codeSearch2=psCompState->cSpecial+2 ; codeSearch2<GIF_LZ_NCODES ; codeSearch2++)
					if (psInfo->abyChainedPixel[codeSearch2]==codeThis && psInfo->acChainedCode[codeSearch2]==cNextEntry)
						{	codeThis=cNextEntry;
							cNextEntry=codeSearch2;
		 					printf(",%03X",cNextEntry);
							break;
						}
				} while (codeSearch2<GIF_LZ_NCODES);
				printf("\n");
			}
#endif
			// Remember chained code, and return
			psCompState->cPrevious = (gif_code_t)uSearch;
			GIF_ASSERT(psCompState->cPrevious<GIF_LZ_NCODES);
			return true;
		}
	}
	// Didn't find, either table full, or it's not in there
	psInfo->byAddPixel=(BYTE)cNewPixel;
	psInfo->cAddCode=psCompState->cPrevious;
	return false;
}

// Add a value pair to the search table
static void gif_private_search_add(gif_private_write_comp_state_t *psCompState)
{	gif_private_search_t * const psInfo=psCompState->pvDataLookup;
#if defined(GIF_LS_DEBUG) && GIF_LS_DEBUG>=5
	printf("Adding code %03X for %03X + %02X\n",psCompState->cNextEntry,psInfo->byAddPixel,psInfo->cAddCode);
#endif

#ifdef GIF_LS_DEBUG
	GIF_ASSERT(psCompState->psCompInfo->bLowMemoryOption);	// Low memory will search, otherwise direct lookup
#endif
	GIF_ASSERT(psCompState->cNextEntry<GIF_LZ_NCODES);
	psInfo->abyChainedPixel[psCompState->cNextEntry]=psInfo->byAddPixel;
	psInfo->acChainedCode[psCompState->cNextEntry]=psInfo->cAddCode;
	psCompState->cNextEntry++;
//if (psCompState->cNextEntry==GIF_LZ_NCODES) printf("L%u:Next=%u\n",__LINE__,psCompState->cNextEntry);
}

typedef struct gif_private_direct_tag
{	unsigned uNEntries;
	DWORD	dwNextEntry;	// At least BYTE + CODE, so 8 + 12 bits
	gif_code_t *pcPrevious;
} gif_private_direct_t;

// Reset the direct lookup table
static void gif_private_direct_reset(gif_private_write_comp_state_t *psCompState)
{	gif_private_direct_t * const psInfo=psCompState->pvDataLookup;
#ifdef GIF_LS_DEBUG
	GIF_ASSERT(!psCompState->psCompInfo->bLowMemoryOption);	// Low memory will search, otherwise direct lookup
#endif
	psInfo->dwNextEntry=(DWORD)psInfo->uNEntries;	// Set illegal value
	// Clear table
	memset(psInfo->pcPrevious,0,sizeof(psInfo->pcPrevious[0])*psInfo->uNEntries);
}

// Look for value pair, returns true if value found, otherwise false
static bool gif_private_direct_find(gif_private_write_comp_state_t *psCompState,gif_code_t cNewPixel)
{	gif_private_direct_t * const psInfo=(gif_private_direct_t *)psCompState->pvDataLookup;
	gif_code_t codeFound;

#ifdef GIF_LS_DEBUG
	GIF_ASSERT(!psCompState->psCompInfo->bLowMemoryOption);	// Low memory will search, otherwise direct lookup
#endif
	GIF_ASSERT(psCompState->cPrevious<GIF_LZ_NCODES);
	GIF_ASSERT(cNewPixel<(1U<<psCompState->psCompData->byBitsPerPixel));

	psInfo->dwNextEntry=(((DWORD)cNewPixel)<<GIF_LZ_BITS)|psCompState->cPrevious;
	GIF_ASSERT(psInfo->dwNextEntry<psInfo->uNEntries);
	codeFound=psInfo->pcPrevious[psInfo->dwNextEntry];
	if (!codeFound)
	{
#if defined(GIF_LS_DEBUG) && GIF_LS_DEBUG>=4
		printf("Chaining %03X followed by %02X (Not found)\n",psCompState->cPrevious,cNewPixel);
#endif
		return false;
	}
#if defined(GIF_LS_DEBUG) && GIF_LS_DEBUG>=4
	printf("Chaining %03X followed by %02X,%03X\n",psCompState->cPrevious,cNewPixel,codeFound);
#endif
	psCompState->cPrevious = codeFound;
	GIF_ASSERT(psCompState->cPrevious<GIF_LZ_NCODES);
	return true;
}

// Add a value pair to the lookup table
static void gif_private_direct_add(gif_private_write_comp_state_t *psCompState)
{	gif_private_direct_t * const psInfo=(gif_private_direct_t *)psCompState->pvDataLookup;
#ifdef GIF_LS_DEBUG
	GIF_ASSERT(!psCompState->psCompInfo->bLowMemoryOption);	// Low memory will search, otherwise direct lookup
#endif
#if defined(GIF_LS_DEBUG) && GIF_LS_DEBUG>=5
	printf("Adding code %03X for %03X + %02X\n",psCompState->cNextEntry,psInfo->dwNextEntry&(GIF_LZ_NCODES-1),psInfo->dwNextEntry>>GIF_LZ_BITS);
#endif
	GIF_ASSERT(psCompState->cNextEntry<GIF_LZ_NCODES);
	GIF_ASSERT(psInfo->dwNextEntry<psInfo->uNEntries);
	psInfo->pcPrevious[psInfo->dwNextEntry]=psCompState->cNextEntry++;
//if (psCompState->cNextEntry==GIF_LZ_NCODES) printf("L%u:Next=%u\n",__LINE__,psCompState->cNextEntry);
}

static void gif_private_search_table_free(gif_private_write_comp_state_t *psCompState,gif_private_write_info_t *psCompInfo)
{	free(psCompState->pvDataLookup);
	psCompState->pvDataLookup=NULL;
	psCompInfo->pfnCodeAdd=NULL;
	psCompInfo->pfnCodeFind=NULL;
	psCompInfo->pfnCodeReset=NULL;
	psCompInfo->pfnSearchFree=NULL;
}

// This version always uses a linear search for the token lookup
static bool gif_private_search_init_lookup(gif_private_write_comp_state_t *psCompState,gif_private_write_info_t *psCompInfo)
{	// Allocate structure for search, and point to linear search functions
	psCompState->pvDataLookup=malloc(sizeof(gif_private_search_t));
	if (!psCompState->pvDataLookup)
		return true;
#ifdef GIF_LS_DEBUG
	psCompInfo->bLowMemoryOption=true;
#endif
	psCompInfo->pfnCodeAdd=gif_private_search_add;
	psCompInfo->pfnCodeFind=gif_private_search_find;
	psCompInfo->pfnCodeReset=gif_private_search_reset;
	psCompInfo->pfnSearchFree=gif_private_search_table_free;
	return false;
}

// This version will try to use a direct mapped lookup, if the allocation for that fails, then falls back to linear search
static bool gif_private_search_init(gif_private_write_comp_state_t *psCompState,gif_private_write_info_t *psCompInfo)
{	unsigned const uDirectEntries=(1U<<(psCompState->psCompData->byBitsPerPixel+GIF_LZ_BITS));
	GIF_ASSERT(psCompState->pvDataLookup==NULL);
	GIF_ASSERT(psCompState->psCompInfo==psCompInfo);
#if 1
	psCompState->pvDataLookup=malloc(sizeof(gif_private_direct_t)+uDirectEntries*sizeof(((gif_private_direct_t *)NULL)->pcPrevious[0]));
	if (psCompState->pvDataLookup)
	{	// The big allocation passed
		gif_private_direct_t *psDirect=(gif_private_direct_t *)psCompState->pvDataLookup;
		psDirect->pcPrevious=(gif_code_t *)(psDirect+1);
		psDirect->uNEntries=uDirectEntries;
		psDirect->dwNextEntry=uDirectEntries;
#ifdef GIF_LS_DEBUG
		psCompInfo->bLowMemoryOption=false;
#endif
		psCompInfo->pfnCodeAdd=gif_private_direct_add;
		psCompInfo->pfnCodeFind=gif_private_direct_find;
		psCompInfo->pfnCodeReset=gif_private_direct_reset;
		psCompInfo->pfnSearchFree=gif_private_search_table_free;
#if defined(GIF_LS_DEBUG) && GIF_LS_DEBUG>=5
		printf("Using direct lookup for %u+%u bits = %u entries\n",psCompState->psCompData->byBitsPerPixel,GIF_LZ_BITS,psDirect->uNEntries);
#endif
		return false;
	}
#endif
	return gif_private_search_init_lookup(psCompState,psCompInfo);
}

// While testing, just add up the number of bits required
static inline void gif_private_write_justcount(gif_private_write_comp_state_t *psCompState)
{	psCompState->uBitsWritten +=psCompState->byNCodeBits;
	if ( psCompState->cNextEntry > psCompState->cMax && psCompState->byNCodeBits<GIF_LZ_BITS)
	{	// The next entry is going to be too big for the code size, so increase code size
		psCompState->byNCodeBits++;
		psCompState->cMax = GIF_PRIVATE_MAXCODE(psCompState->byNCodeBits);
	}
}

// List outcomes that the decision function can return
typedef enum { e_chain, e_overrun, e_reset, e_reset_exit, e_just_exit } gif_private_write_decision_t;

// Returns true if reset required
static gif_private_write_decision_t gif_private_write_image_reset_decision(gif_private_write_comp_state_t *psCodeTableStatus)
{
//if (psCodeTableStatus->psCompData->uNPixels==321114 && psCodeTableStatus->cNextEntry>=4095)
//{	printf("Failing?\n");
//}

	if (psCodeTableStatus->cNextEntry < GIF_LZ_NCODES)
	{	return e_chain;
	}

	GIF_ASSERT(psCodeTableStatus->pvDecisionData!=NULL);


	if (psCodeTableStatus->psCompData->sOptions.bNoOverrun)
	{	return e_reset;
	}

	{	gif_private_write_ratio_info_t *psRatio=(gif_private_write_ratio_info_t *)psCodeTableStatus->pvDecisionData;
#ifdef GIF_PRIVATE_LS_MAGIC
//		if (psRatio->uMagic!=GIF_PRIVATE_LS_MAGIC)
//			printf("At %p (%p), got 0x%X expected 0x%X\n",psRatio,&psRatio->uMagic,psRatio->uMagic,GIF_PRIVATE_LS_MAGIC);
		GIF_ASSERT(psRatio->uMagic==GIF_PRIVATE_LS_MAGIC);
#endif
		if (psRatio->uEndBit==0)
		{	// Just starting overrun, set current values
			psRatio->uEndBit=psCodeTableStatus->uBitsWritten;
			psRatio->uEndPixel=psCodeTableStatus->uPixel;
			return e_overrun;
		} else
		{	unsigned const uStartBit=(psCodeTableStatus->psCompData->psStatus) ? psCodeTableStatus->psCompData->psStatus->uBitsUsed : 0;

#if defined(GIF_LS_DEBUG) && GIF_LS_DEBUG>=2
			{	unsigned const uDiffPixel=psCodeTableStatus->uPixel-psRatio->uEndPixel;
				unsigned const uDiffBits=psCodeTableStatus->uBitsWritten-psRatio->uEndBit;
				unsigned const uBasePixel=psRatio->uEndPixel-psCodeTableStatus->uStartPixel;
				unsigned const uBaseBits=psRatio->uEndBit-uStartBit;
				printf("%u/%u=%.2f vs %u/%u=%.2f vs %u/%u=%.2f\n",
					uDiffBits,uDiffPixel,(double)uDiffBits/(double)uDiffPixel,
					uBaseBits,uBasePixel,(double)uBaseBits/(double)uBasePixel,
					psCodeTableStatus->uBitsWritten-uStartBit,
					psCodeTableStatus->uPixel-psCodeTableStatus->uStartPixel,
					(double)(psCodeTableStatus->uBitsWritten-uStartBit)/(double)(psCodeTableStatus->uPixel-psCodeTableStatus->uStartPixel));
			}
#endif

			// Allow for extra reset code
			// total-pixels / total-bits > original-pixel / original-bits
			//    ===>
			// extra-pixels * original-bits > original-pixel * extra-bits
			if ((psCodeTableStatus->uPixel-psCodeTableStatus->uStartPixel)*(psCodeTableStatus->psCompInfo->byInitialBitsPerCode+psRatio->uEndBit-uStartBit)<=
				(psRatio->uEndPixel-psCodeTableStatus->uStartPixel)*(psCodeTableStatus->uBitsWritten-uStartBit))
			{	// Time to reset
				memset(psRatio,0,sizeof(*psRatio));
#ifdef GIF_PRIVATE_LS_MAGIC
				psRatio->uMagic=GIF_PRIVATE_LS_MAGIC;
//				printf("At %p (%p), set 0x%X\n",psRatio,&psRatio->uMagic,psRatio->uMagic);
#endif
				return e_reset;
			}
			// Current ratio is OK
			return e_overrun;
		}
	}
}

// Write data via output function
static bool gif_private_write_image_data(gif_private_write_data_t const *psCompData,unsigned uStartPixel,BYTE byStartCodeSize,gif_private_write_decision_t (*pfnResetDecision)(gif_private_write_comp_state_t *psState),void *pvDecisionData)
{	gif_private_write_comp_state_t sCodeTableStatus;
	gif_private_write_info_t sCompInfo;
	BYTE const *pbyPixel;

#if 0//defined(GIF_LS_DEBUG) && GIF_LS_DEBUG>=2
	printf("psCompData->psImage = (%u,%u)+(%u,%u) BPP-1=%u/%u, int=%u, Local=%u, %u colour bytes, %u pixels\n",
		psCompData->psImage->image.left,psCompData->psImage->image.top,psCompData->psImage->image.width,psCompData->psImage->image.height,
		psCompData->psImage->image.bitsPerPixelMinus1&7,psCompData->psGifInfo->screen.bitsPerPixelMinus1&7,
		psCompData->psImage->image.bInterlaced&1,psCompData->psImage->image.bLocalColourTable&1,
		psCompData->psImage->uColourTableBytes,psCompData->uNPixels);
	if (psCompData->psImage->bHaveControl)
	{	printf("psCompData->psImage control = trans=%u, disp=%u, delay=%u, trancol=%u\n",
			psCompData->psImage->control.bTransparent&1,psCompData->psImage->control.disposal&7,psCompData->psImage->control.delaytime,psCompData->psImage->control.byTransparentColour);
	}
#endif

	// Sanity checks
	GIF_ASSERT(psCompData && psCompData->pbyPixels && psCompData->uNPixels>0);
	GIF_ASSERT(psCompData->pbyPixels[0]<(1U<<psCompData->byBitsPerPixel));
	GIF_ASSERT(pfnResetDecision!=NULL);

#if defined(GIF_LS_DEBUG) || defined(_DEBUG)
	{	unsigned uPos;
		for (uPos=0 ; uPos<psCompData->uNPixels ; uPos++)
		{	GIF_ASSERT(psCompData->pbyPixels[uPos]<(1U<<psCompData->byBitsPerPixel));
		}
	}
#endif

	// Initialise tables, write code size token
	if (gif_private_write_comp_init(&sCodeTableStatus,&sCompInfo,psCompData,gif_private_search_init))
		return true;

	if (uStartPixel==0)
	{	if (psCompData->pfnWriteData!=NULL &&
			gif_private_write_byte(psCompData, sCompInfo.byInitCodeSize))
		{	// It's so much easier to write this out here, as we have the value available
			gif_private_search_free(&sCodeTableStatus,&sCompInfo);
			return true;
		}
	}

	// Reset overrun info
	sCodeTableStatus.uExtraPixels=0;
	if (psCompData->psStatus)
	{	psCompData->psStatus->uMaxOverrun=0;
		if (uStartPixel)
		{	// If it's not from the beginning, then use the number of initial bits
			sCodeTableStatus.uBitsWritten=psCompData->psStatus->uBitsUsed;
		} else
		{	sCodeTableStatus.uBitsWritten=0;
		}
	}

	sCodeTableStatus.pvDecisionData=pvDecisionData;
	sCodeTableStatus.uStartPixel=sCodeTableStatus.uPixel=uStartPixel;
	if (!psCompData->sOptions.bMissingInitialResets || uStartPixel>0)
	{	if (uStartPixel>0)
		{	GIF_ASSERT(byStartCodeSize>=3 && byStartCodeSize<=GIF_LZ_BITS);
			sCodeTableStatus.byNCodeBits=byStartCodeSize;
		}
		// Output clear code, and reset tables
		if (gif_private_write_comp_clearcode(&sCodeTableStatus))
		{	gif_private_search_free(&sCodeTableStatus,&sCompInfo);
			return true;
		}
	} else
	{	// Don't need to output the clear code, but make sure the structures are reset
		gif_private_write_comp_reset(&sCodeTableStatus);
	}
#if defined(GIF_LS_DEBUG) && GIF_LS_DEBUG>=3
	printf("Pixel data at %p, from pixel %u to %u\n",psCompData->pbyPixels,uStartPixel,uStartPixel+psCompData->uNPixels);
#endif
	for (sCodeTableStatus.uPixel=uStartPixel, pbyPixel=psCompData->pbyPixels+uStartPixel ; sCodeTableStatus.uPixel<psCompData->uNPixels ; sCodeTableStatus.uPixel++)
	{	// Next pixel
		BYTE const byNewValue= (*(pbyPixel++));

		// If first pixel, skip to next
		if (sCodeTableStatus.uPixel==uStartPixel)
		{	sCodeTableStatus.cPrevious=byNewValue;
			continue;
		}

		// Try to chain it
		if (sCompInfo.pfnCodeFind(&sCodeTableStatus,byNewValue))
		{	// It chained
			continue;
		}

		// Didn't chain, write previous code
		if (gif_private_write_code(&sCodeTableStatus, sCodeTableStatus.cPrevious))
		{	gif_private_search_free(&sCodeTableStatus,&sCompInfo);
			return true;
		}

		// Save new code
		sCodeTableStatus.cPrevious=byNewValue;

		// Do we decide to reset
		switch (pfnResetDecision(&sCodeTableStatus))
		{	case e_chain:
				// Store chained pair
				GIF_ASSERT(sCodeTableStatus.cNextEntry < GIF_LZ_NCODES);
				sCompInfo.pfnCodeAdd(&sCodeTableStatus);
				break;
			case e_overrun:
				// Allowed overrun
				GIF_ASSERT(sCodeTableStatus.cNextEntry >= GIF_LZ_NCODES);
				sCodeTableStatus.uExtraPixels++;
				if (psCompData->psStatus)
				{	// Got status block, see if this is the biggest overrun so far
					if (sCodeTableStatus.uExtraPixels>psCompData->psStatus->uMaxOverrun)
					{	psCompData->psStatus->uMaxOverrun=sCodeTableStatus.uExtraPixels;
					}
				}
				break;
			case e_reset:
				// Output clear code, and reset tables
				if (gif_private_write_comp_clearcode(&sCodeTableStatus))
				{	// If error, then bail
					gif_private_search_free(&sCodeTableStatus,&sCompInfo);
					return true;
				}
				// Reset overrun counter
				sCodeTableStatus.uExtraPixels=0;
				break;
			case e_reset_exit:
				// Output clear code, and reset tables
				if (gif_private_write_comp_clearcode(&sCodeTableStatus))
				{	// If error, then bail
					gif_private_search_free(&sCodeTableStatus,&sCompInfo);
					return true;
				}
				// Drop into just exit option
			case e_just_exit:
				// Have we got a status block?
				if (psCompData->psStatus)
				{	// Save the current bits written info
					psCompData->psStatus->uBitsUsed=sCodeTableStatus.uBitsWritten;
				}
				// Bail out early
				gif_private_search_free(&sCodeTableStatus,&sCompInfo);
				return false;
			default:
				GIF_ASSERT("Shouldn't get here!");
		}
	}

	// Output final code, then EOI
	if (gif_private_write_code(&sCodeTableStatus, (gif_code_t)(sCodeTableStatus.cPrevious)) ||
		gif_private_write_comp_eoi(&sCodeTableStatus))
	{	gif_private_search_free(&sCodeTableStatus,&sCompInfo);
		return true;
	}

	// Have we got a status block?
	if (psCompData->psStatus)
	{	// Save the current bits written info
		psCompData->psStatus->uBitsUsed=sCodeTableStatus.uBitsWritten;
	}

	gif_private_search_free(&sCodeTableStatus,&sCompInfo);
	return false;
}

static bool gif_private_write_image_data_basic(gif_private_write_data_t *psCompData)
{	gif_private_write_ratio_info_t sRatio;
	memset(&sRatio,0,sizeof(sRatio));
#ifdef GIF_PRIVATE_LS_MAGIC
	sRatio.uMagic=GIF_PRIVATE_LS_MAGIC;
//	printf("At %p (%p), set 0x%X\n",&sRatio,&sRatio.uMagic,sRatio.uMagic);
#endif
	return gif_private_write_image_data(psCompData,0,0,gif_private_write_image_reset_decision,&sRatio);
}

// Write out a colour map
static bool gif_private_write_colourmap(gif_private_write_data_t *psCompData,gif_colour_entry_t const *pColour,unsigned uNColours)
{	unsigned uColour;
	for( uColour=0 ; uColour<uNColours ; uColour++,pColour++ )
	{	if (gif_private_write_byte(psCompData, pColour->red) ||
			gif_private_write_byte(psCompData, pColour->green) ||
			gif_private_write_byte(psCompData, pColour->blue))
				return true;
	}
	return false;
}

// Write a single compressed GIF image as part of a file
static bool gif_private_write_image(gif_file_t const *psGifInfo,gif_private_write_data_t *psCompData,gif_image_t *psImage,bool (*pfnWriteImage)(gif_private_write_data_t *psCompData))
{	gif_private_write_status_t sStatus;
	BYTE byTemp;

	// Sanity check input params
	GIF_ASSERT((unsigned)(psImage->image.height*psImage->image.width)==psImage->uNPixels);
	GIF_ASSERT((psImage->image.left+psImage->image.width)<=psGifInfo->screen.width);
	GIF_ASSERT((psImage->image.top+psImage->image.height)<=psGifInfo->screen.height);

	// Must have a colour map somewhere
	GIF_ASSERT(psImage->image.bLocalColourTable || psGifInfo->screen.bGlobalColourTable);
	GIF_ASSERT(psImage->pLines);
	GIF_ASSERT(pfnWriteImage);

	// If we've a netscape loop, then deal with it
	if (psImage->bHaveNetscapeLoop)
	{	gif_ext_application_t block;
		memcpy_s(&block,sizeof(block),GIF_NETSCAPE_ID,sizeof(block));
		if (gif_private_write_byte(psCompData, 0x21) ||
			gif_private_write_byte(psCompData, 0xFF) ||
			gif_private_write_byte(psCompData, (BYTE)sizeof(block)) ||
			psCompData->pfnWriteData((BYTE *)&block, sizeof(block), psCompData->pvDataHandle) ||
			gif_private_write_byte(psCompData, (BYTE)3) ||
			gif_private_write_byte(psCompData, (BYTE)1) ||
			gif_private_write_word(psCompData, psImage->wNumLoops) ||
			gif_private_write_byte(psCompData, 0x00))
				return true;
	}

	// If we've a control block, then deal with it
	if (psImage->bHaveControl)
	{	byTemp=(BYTE)(psImage->control.disposal<<2);
		if (psImage->control.bTransparent)
			byTemp|=1;
		if (psImage->control.bUserinput)
			byTemp|=2;
		if (gif_private_write_byte(psCompData, 0x21) ||
			gif_private_write_byte(psCompData, 0xF9) ||
			gif_private_write_byte(psCompData, (BYTE)sizeof(psImage->control)) ||
			gif_private_write_byte(psCompData, byTemp) ||
			gif_private_write_word(psCompData, psImage->control.delaytime) ||
			gif_private_write_byte(psCompData, psImage->control.byTransparentColour) ||
			gif_private_write_byte(psCompData, 0x00))
			return true;
	}

	// Write an Image separator
	if (gif_private_write_byte(psCompData, ','))
		return true;

	// Write the Image header
	if (gif_private_write_word(psCompData, psImage->image.left) ||
		gif_private_write_word(psCompData, psImage->image.top) ||
		gif_private_write_word(psCompData, psImage->image.width) ||
		gif_private_write_word(psCompData, psImage->image.height))
		return true;

	// Write out image flags
	byTemp=(BYTE)psImage->image.bitsPerPixelMinus1;
	if (psImage->image.bLocalColourTable)
		byTemp|=0x80;
	if (psImage->image.bInterlaced)
		byTemp|=0x40;
	if (gif_private_write_byte(psCompData, byTemp))
		return true;

	// Sort out colour maps, and depth, this sets initial code size
	if (psImage->image.bLocalColourTable)
	{	// If we have a Local Colour Map
		psCompData->byBitsPerPixel=(BYTE)(psImage->image.bitsPerPixelMinus1+1);
		if (gif_private_write_colourmap(psCompData,psImage->psColourTable,GIF_NCOLOURS(psImage->image)))
			return true;
	} else
	{	// Otherwise use screen info
		psCompData->byBitsPerPixel=(BYTE)(psGifInfo->screen.bitsPerPixelMinus1+1);
	}

#if 0//defined(GIF_LS_DEBUG) && GIF_LS_DEBUG>=3
	printf("Image %ux%u:%u%c = %u pixels, %u bpp = %u bits (uncompresses)\n",
		psCompData->psImage->image.width,psCompData->psImage->image.height,1<<psCompData->byBitsPerPixel,
		(psCompData->psImage->image.bLocalColourTable) ? 'L' : 'G',
		psCompData->uNPixels,psCompData->byBitsPerPixel,psCompData->uNPixels*psCompData->byBitsPerPixel);
#endif

	// Point image info to current image
	psCompData->psStatus	= &sStatus;
	psCompData->pbyPixels	= psImage->pLines[0];
	psCompData->uNPixels	= psImage->uNPixels;

	if (psCompData->uNPixels==0)
	{	// Output empty image (Just a start code, code size and a wrapped EOI)
		gif_private_write_comp_state_t sEmptyState;
		gif_private_write_info_t sEmptyInfo;
		if (gif_private_write_comp_init(&sEmptyState,&sEmptyInfo,psCompData,gif_private_search_init_lookup))
			return true;
		if (gif_private_write_byte(psCompData, sEmptyInfo.byInitCodeSize) ||
			gif_private_write_comp_eoi(&sEmptyState))
		{	gif_private_search_free(&sEmptyState,&sEmptyInfo);
			return true;
		}
		gif_private_search_free(&sEmptyState,&sEmptyInfo);
#if defined(GIF_LS_DEBUG) || defined(GIF_LS_STATS) || defined(GIF_OPT_DEBUG)
		{	// Display post compression info
			printf("Image %u pixels\n",psCompData->uNPixels);
		}
#endif
	} else
	{	// Output full image, using specified write function (probably gif_private_write_image_data_basic)
		if (pfnWriteImage(psCompData))
			return true;
#if defined(GIF_LS_DEBUG) || defined(GIF_LS_STATS) || defined(GIF_OPT_DEBUG)
		{	unsigned uColours;
			unsigned uPixel;
			unsigned uNBits;
			BYTE const *pbyPixel;
			bool abColourUsed[256];

			// Work out number of colours actually used
			memset(abColourUsed,0,sizeof(abColourUsed));
			uColours=0;
			for (uPixel=0, pbyPixel=psCompData->pbyPixels ; uPixel<psCompData->uNPixels ; uPixel++,pbyPixel++)
			if (!abColourUsed[*pbyPixel])
			{	abColourUsed[*pbyPixel]=true;
				uColours++;
			}

			// Work out the number of bits actually required
			for (uNBits=1 ; uNBits<=8 ; uNBits++)
				if ((1U<<uNBits)>=uColours)
					break;

			// If local should match bits per pixel value
			//GIF_ASSERT(!psCompData->psImage->image.bLocalColourTable || uNBits==psCompData->byBitsPerPixel);

			// Display post compression info
			printf("Image %u pixels, %u(%u) bpp used %u bits (%u%%)\n",
				psCompData->uNPixels,psCompData->byBitsPerPixel,uNBits,psCompData->psStatus->uBitsUsed,100*psCompData->psStatus->uBitsUsed/(psCompData->uNPixels*uNBits));
		}
#endif
	}

	// Write out an Image terminator
	if (gif_private_write_byte(psCompData, 0))
		return true;

	psCompData->psStatus=NULL;
	return false;
}

// Write bytes to a FILE, void *pvDataHandle==FILE * of file (returns true on error)
static bool gif_private_write_data_to_file(BYTE const *pSrc,unsigned uNBytes,void *pvDataHandle)
{	if (fwrite(pSrc,uNBytes,1,(FILE *)pvDataHandle)==1) return false;
	fprintf(stderr,"\nError: Failed to write %u bytes to file\n",uNBytes);
	return true;
}

// Write a gif file, using supplied write_function
bool gif_private_write_images(gif_file_t const *psGifInfo,gif_options_t const *psOptions,
			bool	(*pfnWriteData)(const BYTE *pDest,unsigned uNBytes,void *pvDataHandle),
			bool	(*pfnWriteCode)(gif_private_write_comp_state_t *psInfo, gif_code_t code),
			bool	(*pfnWriteImage)(gif_private_write_data_t *psCompData),
			void	*pvDataHandle)
{	gif_private_write_data_t sComp;
#ifdef GIF_LS_STATS
	time_t sStart,sEnd;
	time(&sStart);
#endif
	if (!psGifInfo || psGifInfo->uNImages==0 || !psGifInfo->psImages || !pfnWriteImage)
		return true;

	// Point private pointer to illegal value so that special functions will fail if called too early
	((gif_file_t *)psGifInfo)->pPrivate=NULL; // (void *)&sComp;

	// Setup pointers we'll need
	memset(&sComp,0,sizeof(sComp));
	sComp.pfnWriteData=pfnWriteData;
	sComp.pfnWriteCode=pfnWriteCode;
	sComp.pvDataHandle=pvDataHandle;

	if (psOptions)
	{	sComp.sOptions= *psOptions;
	} else
	{	// all 0 = reasonable defaults
		memset(&sComp.sOptions,0,sizeof(sComp.sOptions));
	}

	// Write the ID string
	if (sComp.pfnWriteData((BYTE*)(gif_is_gif89(psGifInfo) ? "GIF89a" : "GIF87a"), 6, pvDataHandle))
		return true;

	// Write out the screen width and height
	if (gif_private_write_word(&sComp, psGifInfo->screen.width) ||
		gif_private_write_word(&sComp, psGifInfo->screen.height))
		return true;

	// Write out flag values
	if (gif_private_write_byte(&sComp,
			(BYTE)(((psGifInfo->screen.bGlobalColourTable) ? 0x80 : 0) |	// Indicate if there is a global colour map
					(psGifInfo->screen.bitsPerColourValMinus1 << 4) |	// Colour definition (actually number of significant bits per colour value)
					psGifInfo->screen.bitsPerPixelMinus1) ) )			// The Bits per Pixel
		return true;

	// Write out the Background colour
	if (gif_private_write_byte(&sComp, psGifInfo->screen.bgColour)) return true;

	// In GIF87a this byte is for future expansion, In GIF89a it is the aspect ratio, 0 = 1:1
	if (gif_private_write_byte(&sComp, 0)) return true;

	// If we have a global colour map write it out
	if (psGifInfo->screen.bGlobalColourTable &&
		gif_private_write_colourmap(&sComp,psGifInfo->psColourTable,GIF_NCOLOURS(psGifInfo->screen)))
		return true;

	// Write all the images
	{	unsigned uImage;
		gif_image_t *psImage;

#ifndef GIF_LS_NOSTATUS
		sComp.uPixelsTotal=0;
		sComp.uPixelsDone=0;
		for (uImage=0, psImage= psGifInfo->psImages; uImage<psGifInfo->uNImages ; uImage++,psImage++)
		{	GIF_ASSERT(psImage->uNPixels==(unsigned)(psImage->image.width*psImage->image.height));
			sComp.uPixelsTotal+=psImage->uNPixels;
		}
#endif

		for (uImage=0, psImage= psGifInfo->psImages; uImage<psGifInfo->uNImages ; uImage++,psImage++)
		{	//printf("Image %u starts at 0x%X\n",uImage+1,ftell((FILE *)pvDataHandle));
#ifdef MEMORY_DEBUG
//			memdebug_check_all();
#endif
			if (gif_private_write_image(psGifInfo,&sComp,psImage,pfnWriteImage))
				return true;
#ifndef GIF_LS_NOSTATUS
			sComp.uPixelsDone+=psImage->uNPixels;
#endif
		}
#ifndef GIF_LS_NOSTATUS
		GIF_ASSERT(sComp.uPixelsDone==sComp.uPixelsTotal);
#endif
	}

	// Write the GIF file terminator
	if (gif_private_write_byte(&sComp, ';')) return true;
#ifdef GIF_LS_STATS
	time(&sEnd);
	printf("Write took %u seconds\n",(unsigned)(sEnd-sStart));
#endif
	return false;
}

/************************** Public functions *******************************/

void gif_clear_unused_colours(gif_colour_entry_t *psColours,unsigned const uNColours,bool const *pbUsedList)
{	unsigned uClear;
	for (uClear=0 ; uClear<uNColours ; uClear++)
	if (!pbUsedList[uClear])
	{	psColours[uClear].red=psColours[uClear].green=psColours[uClear].blue=0xFF;
	}
}

// Write a gif file, using supplied write_function
bool gif_write_images(gif_file_t const *psGifInfo,gif_options_t const *psOptions,
			bool	(*pfnWriteData)(const BYTE *pDest,unsigned uNBytes,void *pvDataHandle),
			void	*pvDataHandle)
{	return gif_private_write_images(psGifInfo,psOptions,pfnWriteData,NULL,gif_private_write_image_data_basic,pvDataHandle);
}

// Write a gif file, to a file with given name
bool gif_write_file(const gif_file_t *pInfo,gif_options_t const *psOptions,const char *pFileName)
{	FILE *out=fopen(pFileName,"wb");
	bool bRetVal;
	if (!out)
	{	fprintf(stderr,"\nError: Failed to open '%s'\n",pFileName);
		return true;
	}
	bRetVal=gif_write_images(pInfo,psOptions,gif_private_write_data_to_file,out);
	fclose(out);
	return bRetVal;
}

#ifdef GIF_PRIVATE_WRITE_TO_MEM
// Write pInfo to block of memory, return *pData pointing to memory, *puLength=length
bool gif_write_memory(const gif_file_t *pInfo,gif_options_t const *psOptions,char **pData,unsigned *puLength)
{	gif_private_write_mem_t sMemInfo;
	if (pData==NULL || puLength==NULL)
		return true;
	memset(&sMemInfo,0,sizeof(sMemInfo));
	if (gif_write_images(pInfo,psOptions,gif_private_write_data_to_mem,&sMemInfo))
	{	free(sMemInfo.pData);
		*pData=NULL;
		*puLength=0;
		return true;
	}
	*pData=(char *)sMemInfo.pData;
	*puLength=sMemInfo.uLen;
	return false;
}
#endif

// Copy screen and global colour table to new image
bool gif_copy_image_info(gif_file_t *pDest,gif_file_t const *pSrc)
{	// Copy screen info etc
	memset(pDest,0,sizeof(*pDest));
	pDest->screen=pSrc->screen;
	if (pSrc->screen.bGlobalColourTable)
	{	const unsigned uEntries=1<<(pSrc->screen.bitsPerPixelMinus1+1);
		const unsigned uNBytes=uEntries*sizeof(*pDest->psColourTable);
		if ((pDest->psColourTable=malloc(uNBytes))==NULL)
		{	fprintf(stderr,"\nError: Out of memory\n");
			return true;
		}
		pDest->uColourTableBytes=uNBytes;
		memcpy_s(pDest->psColourTable,uNBytes,pSrc->psColourTable,uNBytes);
	}
	return false;
}

#endif // NO_FUNCTIONS ]

#ifdef __cplusplus
}
#endif

#endif // DONE_GIF_LS_H ]
